<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @package    Alfa_Tools
 * @subpackage Alfa_Tools/public
 */
class Alfa_Tools_Public {

	private $alfa_tools;
	private $version;

	public function __construct( $alfa_tools, $version ) {
		$this->alfa_tools = $alfa_tools;
		$this->version = $version;

		add_action('rest_api_init', function () {
			register_rest_route('alfa-tools/v1', '/get_phone_link/', [
				'methods'  => 'POST',
				'callback' => [$this, 'get_phone_link'],
				'args'     => [
					'nonce' => [
						'type'     => 'string',
						'required' => true,
					],
				],
			]);
		});

		add_action('wp_footer', [$this, 'render_whatsapp_button']);
	}

	public function get_phone_link(WP_REST_Request $request) {
		session_start();

		if (
			!isset($_SESSION['phone']) || 
			!isset($_SESSION['phone_date']) || 
			$_SESSION['phone_date'] < (time() - 120)
		) {
			$_SESSION['phone_date'] = time();

			if (is_multisite()) {
				switch_to_blog(1);
				$secret = get_option('_alfa_tools_secret_key');
				restore_current_blog();
			} else {
				$secret = get_option('_alfa_tools_secret_key');
			}

			$url = "https://alfa-phone.com/api/phones/free_number";

			$args = [
				'headers' => [
					'Content-Type' => 'application/json',
					'Accept'       => 'application/json',
				],
				'body'    => wp_json_encode([
					'secret_key' => $secret,
					'domain'     => get_site_url(),
				]),
				'timeout' => 10,
				'sslverify' => false, // можно убрать на продакшене
			];

			$response = wp_remote_post($url, $args);

			if (is_wp_error($response)) {
				return ['error' => $response->get_error_message()];
			}

			$body = wp_remote_retrieve_body($response);
			$_SESSION['phone'] = $body;
			$result = json_decode($body);
		} else {
			$result = json_decode($_SESSION['phone']);
		}

		return $result;
	}

	public function get_whatsapp() {
		$cache_key = 'alfa_tools_whatsapp_data';
		$cached = wp_cache_get($cache_key, 'alfa_tools');

		if ($cached !== false) {
			return json_decode($cached);
		}

		if (is_multisite()) {
			switch_to_blog(1);
			$secret = get_option('_alfa_tools_secret_key');
			restore_current_blog();
		} else {
			$secret = get_option('_alfa_tools_secret_key');
		}

        $url = add_query_arg(['secret_key' => $secret], 'https://alfa-phone.com/api/v1/whatsapp');

        $args = [
            'headers'    => [
                'Accept' => 'application/json',
            ],
            'timeout'    => 10,
            'sslverify'  => false, // на продакшене лучше true
        ];

        $response = wp_remote_get($url, $args);

		if (is_wp_error($response)) {
			return [
				'error' => $response->get_error_message(),
			];
		}

		$body = wp_remote_retrieve_body($response);

		// Кэшируем на 1 час
		wp_cache_set($cache_key, $body, 'alfa_tools', HOUR_IN_SECONDS);

		return json_decode($body);
	}


	public function enqueue_styles() {
		wp_enqueue_style($this->alfa_tools, plugin_dir_url(__FILE__) . 'css/alfa-tools-public.css', [], $this->version, 'all');
	}

	public function enqueue_scripts() {
		wp_enqueue_script($this->alfa_tools, plugin_dir_url(__FILE__) . 'js/alfa-tools-public.js', ['jquery', 'wp-api'], $this->version, false);
		wp_localize_script($this->alfa_tools, 'REST_API_data', [
			'root'  => esc_url_raw(rest_url()),
			'nonce' => wp_create_nonce('alfa-tools'),
		]);
	}

	public function render_whatsapp_button() {
		$whatsapp_data = $this->get_whatsapp();
		// Если вернулась ошибка или нет plainId
		if (isset($whatsapp_data->error) || empty($whatsapp_data->plainId)) {
			return;
		}

		$whatsapp_number = $whatsapp_data->plainId;
		$phone_data_attribute = 'data-phone="' . esc_attr($whatsapp_number) . '"';
		if (is_multisite()) {
				switch_to_blog(1);
		}
		// Получаем настройки позиционирования
		$top = get_option('_alfa_tools_whatsapp_top', '');
		$left = get_option('_alfa_tools_whatsapp_left', '');
		$right = get_option('_alfa_tools_whatsapp_right', '');
		$bottom = get_option('_alfa_tools_whatsapp_bottom', '');
		if (is_multisite()) {
				restore_current_blog();
		}
		// Формируем CSS только если есть хотя бы одна настройка
		$custom_style = '';
		if ($top) $custom_style .= "top:{$top};";
		if ($left) $custom_style .= "left:{$left};";
		if ($right) $custom_style .= "right:{$right};";
		if ($bottom) $custom_style .= "bottom:{$bottom};";
		?>
		<a id="custom-whatsapp-icon" <?php echo $phone_data_attribute; ?> class="whatsapp-pulsating-icon" href="javascript:void(0);" aria-label="WhatsApp">
			<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 46 46" fill="none" aria-hidden="true" focusable="false">
				<path d="M34.9046 11.0658C31.7461 7.92972 27.5456 6.20177 23.0701 6.20001C13.8489 6.20001 6.34398 13.6424 6.34029 22.7903C6.3391 25.7145 7.10946 28.5688 8.57344 31.0848L6.20001 39.6825L15.0687 37.3753C17.5123 38.697 20.2634 39.3936 23.0635 39.3948H23.0704C32.2906 39.3948 39.7963 31.9515 39.8 22.8035C39.8018 18.3703 38.0633 14.2018 34.9046 11.0658ZM23.0704 36.5927H23.0647C20.5695 36.5917 18.1223 35.927 15.9874 34.6705L15.4796 34.3717L10.2168 35.7408L11.6215 30.6521L11.2908 30.1303C9.89889 27.9347 9.16375 25.397 9.16484 22.7914C9.16785 15.1881 15.4057 9.00218 23.0757 9.00218C26.7897 9.00358 30.281 10.4399 32.9064 13.0464C35.5317 15.653 36.9767 19.1177 36.9753 22.8025C36.9723 30.4064 30.7345 36.5927 23.0704 36.5927ZM30.6976 26.2646C30.2795 26.0571 28.2244 25.0542 27.8412 24.9158C27.458 24.7774 27.1794 24.7083 26.9006 25.1234C26.622 25.5384 25.8209 26.4722 25.577 26.7488C25.3332 27.0255 25.0893 27.0601 24.6713 26.8526C24.2533 26.6451 22.9064 26.2073 21.3097 24.795C20.0671 23.6959 19.2281 22.3382 18.9843 21.9231C18.7404 21.508 18.9584 21.2837 19.1676 21.077C19.3556 20.8913 19.5856 20.5928 19.7946 20.3507C20.0036 20.1087 20.0733 19.9357 20.2126 19.6591C20.352 19.3824 20.2823 19.1402 20.1778 18.9328C20.0733 18.7253 19.2373 16.6847 18.8889 15.8547C18.5496 15.0463 18.2049 15.1557 17.9484 15.143C17.7048 15.131 17.4258 15.1284 17.1472 15.1284C16.8686 15.1284 16.4157 15.2322 16.0325 15.6472C15.6494 16.0622 14.5695 17.0652 14.5695 19.1057C14.5695 21.1462 16.0673 23.1175 16.2764 23.3941C16.4854 23.6708 19.224 27.8582 23.4173 29.6539C24.4147 30.081 25.1933 30.3361 25.8004 30.5272C26.8018 30.8426 27.713 30.7981 28.4334 30.6914C29.2365 30.5724 30.9065 29.6886 31.2549 28.7202C31.6033 27.7518 31.6033 26.9218 31.4987 26.7489C31.3942 26.5759 31.1156 26.4722 30.6976 26.2646Z" fill="white"></path>
			</svg>
		</a>
		<?php if ($custom_style): ?>
		<style>
		#custom-whatsapp-icon.whatsapp-pulsating-icon {
			<?php echo esc_html($custom_style); ?>
		}
		</style>
		<?php endif; ?>
		<?php
	}

}
